package org.goplanit.utils.network.layer.service;

import java.util.Collection;

import org.goplanit.utils.graph.Edge;
import org.goplanit.utils.graph.directed.DirectedVertex;
import org.goplanit.utils.graph.directed.EdgeSegment;
import org.goplanit.utils.network.layer.physical.Node;

/**
 * Service node is a vertex but not all vertices are service nodes.
 * A service node represents a location in the service network where legs intersect, usually
 * representing for example stop locations of public transport.
 * 
 * @author markr
 *
 */
public interface ServiceNode extends DirectedVertex {

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract ServiceNode shallowClone();

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract ServiceNode deepClone();
     
  /**
   * Identical to {@link #getEdges()}
   * 
   * @return legs 
   */
  public default  Collection<? extends Edge> getLegs() {
    return getEdges();
  }
  
  /**
   * Identical to {@link #getEntryEdgeSegments()}
   * 
   * @return entryLegSegments
   */
 
  public default  Iterable<? extends EdgeSegment> getEntryLegSegments() {
    return getEntryEdgeSegments();
  }
  
  /**
   * Identical to {@link #getExitEdgeSegments()}
   * 
   * @return exitLegSegments
   */
 
  public default Iterable<? extends EdgeSegment> getExitLegSegments() {
    return getExitEdgeSegments();
  }


  /** Identical to {@code #getEdgeSegment(DirectedVertex)}
   * 
   * @param endNode to use
   * @return first leg segment matching this signature
   */
  public default ServiceLegSegment getLegSegment(ServiceNode endNode) {
    return (ServiceLegSegment) getEdgeSegment(endNode);
  }


  /**
   * Collect the first available entry leg segment using the iterator internally. It is assumed
   * at least one entry is available
   * 
   * @return first entry available
   */
  public default ServiceLegSegment getFirstEntryLegSegment(){
    return (ServiceLegSegment) getEntryLegSegments().iterator().next();
  }
  
  /**
   * Collect the first available exit link segment using the iterator internally. It is assumed
   * at least one entry is available
   * 
   * @return first exit available
   */
  public default ServiceLegSegment getFirstExitLegSegment(){
    return (ServiceLegSegment) getExitLegSegments().iterator().next();
  }  
  
  /**
   * Provide access to the underlying network node(s)
   * 
   * @return networkLayerNode(s)
   */
  public abstract Collection<Node> getPhysicalParentNodes();

  /**
   * Verify if a physical parent node is linked to this service node
   *
   * @return true when present false otherwise
   */
  public abstract boolean hasPhysicalParentNodes();

  /**
   * Verify if provided node is registered as a physical parent node of this service node
   *
   * @param physicalParentNode to verify
   * @return true when registered, false otherwise
   */
  public abstract boolean isMappedToPhysicalParentNode(Node physicalParentNode);
}
